/** @file   BadAssAlien.cpp
 * @brief   Implementation of BadAssAlien class.
 * @version $Revision: 1.6 $
 * @date    $Date: 2006/07/13 16:28:35 $
 * @author  Tomi Lamminsaari
 */

#include "BadAssAlien.h"
#include "eng2d.h"
#include "Settings.h"
#include "WarGlobals.h"
#include "www_map.h"
#include "GameAnims.h"
#include "AnimPlayer.h"
#include "BadAssAi.h"
#include "AnimId.h"
#include "SoundSamples.h"
using eng2d::Vec2D;
using eng2d::ParticleBlood;
using eng2d::Color;
using eng2d::Animation;
using eng2d::Sound;

namespace WeWantWar {

///
/// Constants, datatypes and static methods
/// ============================================================================

const int KGun1Nose = 0;
const int KGun2Nose = 1;
const int KSecondaryBulletCount = 25;
const float KSecondaryBulletRange = 140.0;

///
/// Constructors, destructor and operators
/// ============================================================================

/** Default constructor.
 */
BadAssAlien::BadAssAlien() :
  Alien()
{
  ObjectID::Type oid = ObjectID::TYPE_BADASSALIEN;
  iWalkVector.vy = -Settings::floatObjProp(oid, "spd_walk:");
  this->boundingSphere( Settings::floatObjProp(oid, "bounding_sphere:") );
  this->setArmor( Settings::floatObjProp(oid, "armor:") );
  this->setProperties( 0 );
  
  m_animation = GameAnims::findAnimation( AnimId::KBadAssAlien, GameAnims::EIdle );
  
  // Set the collision points
  float collisionX = Settings::floatObjProp(oid, "collision_x:");
  float collisionY = Settings::floatObjProp(oid, "collision_y:");
  
  this->setCollisionPoint( 0, Vec2D( -collisionX,-collisionY ) );
  this->setCollisionPoint( 1, Vec2D(  collisionX,-collisionY ) );
  this->setCollisionPoint( 2, Vec2D(  collisionX, collisionY ) );
  this->setCollisionPoint( 3, Vec2D( -collisionX, collisionY ) );
  
  Vec2D gun1Vec( Settings::floatObjProp(oid, "gun_1_x:"),
                 Settings::floatObjProp(oid, "gun_1_y:") );
  Vec2D gun2Vec( Settings::floatObjProp(oid, "gun_2_x:"),
                 Settings::floatObjProp(oid, "gun_2_y:") );
  this->addCtrlPoint( gun1Vec );
  this->addCtrlPoint( gun2Vec );
  
  BadAssAi* controller = new BadAssAi(this);
  controller->setVisualRange( Settings::floatObjProp(oid, "visual_distance:") );
  controller->setAttackDistance( Settings::floatObjProp(oid, "attack_distance:") );
  this->setController( controller );
}



/** Destructor.
 */
BadAssAlien::~BadAssAlien()
{
}



///
/// Methods inhertited from the base class(es)
/// ============================================================================

void BadAssAlien::update()
{
  if ( this->state() == GameObject::STATE_KILLED ) {
    return;
  }
  
  if ( this->state() == GameObject::STATE_DYING ) {
    if ( m_animation.paused() == true ) {
      this->hidden( true );
      this->state( STATE_KILLED );
      this->setCorrectAnimation( GameAnims::EKilled );
      const Animation& explosion = GameAnims::findAnimation( AnimId::KExplosionGrenade );
      AnimPlayer::spawn( explosion, this->position(), 0 );
    }
    return;
  }
  
  m_pController->update();
  if ( m_pController->forward() != 0 ) {
    Vec2D mvec( iWalkVector );
    mvec.rotate( this->angle() );
    this->move( mvec );
    this->setCorrectAnimation( GameAnims::EWalk ); 
  }
  this->changeAngle( m_pController->turn() );
  if ( m_pController->shoot() == 1 ) {
    float secondaryDistance = Settings::floatObjProp(ObjectID::TYPE_BADASSALIEN,
                                                     "secondary_attack:");
    GameObject* player = WarGlobals::pObjTable->pPlayer;
    Vec2D distV = player->position() - this->position();  
    if ( distV.length() < secondaryDistance ) {
      this->attack2(); 
    } else {
      this->attack();
    }    
  }
  if ( m_pController->forward() == 0 && m_pController->shoot() == 0 ) {
    this->setCorrectAnimation( GameAnims::EIdle );
  }
}

void BadAssAlien::makeSound( GameObject::SoundID aSoundId ) const
{
  int soundIndex = -1;
  switch ( aSoundId ) {
    case ( GameObject::SND_DIE ): {
      soundIndex = SMP_BADASSALIEN_DIE;
      break;
    }
    case ( GameObject::SND_PAIN ): {
      soundIndex = SMP_BADASSALIEN_PAIN;
      break;
    }
    case ( GameObject::SND_ATTACK ): {
      soundIndex = SMP_LIGHTBALLGUN;
      break;
    }
    default: {
      soundIndex = -1;
      break;
    }
  }
  if ( soundIndex != - 1 ) {
    Sound::playSample( soundIndex, false );
  }
}

bool BadAssAlien::hitByBullet( Bullet* aBullet )
{
  bool ret = Alien::hitByBullet( aBullet );
  if ( ret == true ) {
    this->makeSound( GameObject::SND_PAIN );
    if ( aBullet->iType != Bullet::EFlameThrower ) {
      ParticleBlood* pP = new ParticleBlood( aBullet->iPosition,
                                             aBullet->velocity(), 12,
                                             Color(130,60,10) );
      WarGlobals::pPartManager->addSystem( pP );
    }
  }

  // Inform the AI-controller that we got hit.
  AIController* pC = dynamic_cast<AIController*>( this->getController() );
  AIController::BFlag f = pC->getFlags();
  f &= ~AIController::WAIT_TARGET;
  pC->setFlags( f );
  return ret;
}

bool BadAssAlien::causeDamage( Bullet* pB )
{
  // We inform the AI-controller.
  BadAssAi* pC = dynamic_cast<BadAssAi*>( m_pController );
  pC->setAlerted( true );
  
  // And let the parent class handle the damage.
  return Alien::causeDamage( pB );
}

void BadAssAlien::kill()
{
  this->state( GameObject::STATE_DYING );
  this->setCorrectAnimation( GameAnims::EDying );
  this->makeSound( GameObject::SND_DIE );
}

ObjectID::Type BadAssAlien::objectType() const
{
  return ObjectID::TYPE_BADASSALIEN;
}


///
/// New public methods
/// ============================================================================




///
/// Getter methods
/// ============================================================================




///
/// Protected and private methods
/// ============================================================================

void BadAssAlien::setCorrectAnimation( int aAnimId )
{
  if ( this->getAnimID() != aAnimId ) {
    const Animation& anim = GameAnims::findAnimation( AnimId::KBadAssAlien, aAnimId );
    this->setAnimation( anim, aAnimId );
  }
}

void BadAssAlien::attack()
{
  if ( this->reloading() == true ) {
    return;
  }
  
  Vec2D gunV1 = this->position() + this->getCtrlPoint( KGun1Nose );
  Vec2D gunV2 = this->position() + this->getCtrlPoint( KGun2Nose );
  Bullet* bullet1 = BulletTable::createBullet(this, gunV1, Bullet::EAlienFireball2 );
  Bullet* bullet2 = BulletTable::createBullet(this, gunV2, Bullet::EAlienFireball2 );
  bullet1->iAnimation = GameAnims::findAnimation( AnimId::KBulletAlienFireball2 );
  bullet2->iAnimation = GameAnims::findAnimation( AnimId::KBulletAlienFireball2 );
  WarGlobals::pBulletManager->spawnBullet( bullet1 );
  WarGlobals::pBulletManager->spawnBullet( bullet2 );
  this->makeSound( GameObject::SND_ATTACK );
  this->setCounter( Alien::RELOAD_COUNTER_INDEX, Settings::intObjProp(ObjectID::TYPE_BADASSALIEN, "reload:") );
  this->setCorrectAnimation( GameAnims::EShoot );
  
  const Animation& flameAnim = GameAnims::findAnimation( AnimId::KRifleShootFlame );
  AnimPlayer::spawn( flameAnim, gunV1, 0 );
  AnimPlayer::spawn( flameAnim, gunV2, 0 );
  const Animation& lightAnim = GameAnims::findAnimation( AnimId::KRifleShotLight );
  if ( Settings::shootingLights == true ) {
    AnimPlayer::spawn( lightAnim, gunV1, 0 );
    AnimPlayer::spawn( lightAnim, gunV2, 0 );
  }
  
  this->makeSound( GameObject::SND_ATTACK );
}

void BadAssAlien::attack2()
{
  if ( this->reloading() == true ) {
    return;
  }
  ObjectID::Type oid = ObjectID::TYPE_BADASSALIEN;
  int reloadDelay = Settings::intObjProp(oid, "reload2:");
  this->setCounter ( Alien::RELOAD_COUNTER_INDEX, reloadDelay );
  float angleStep = 256 / static_cast<float>( KSecondaryBulletCount );
  float shootAngle = 0;
  for ( int i=0; i < KSecondaryBulletCount; i++ ) {
    Bullet* pB = BulletTable::createBullet( this, m_position, Bullet::EAlienFireball2 );
    Vec2D spdV( pB->velocity() );
    spdV.rotate( shootAngle );
    pB->setVelocity( spdV );
    pB->iRange = KSecondaryBulletRange;
    WarGlobals::pBulletManager->spawnBullet( pB );
    shootAngle += angleStep;
  }
  Sound::playSample( SMP_TANKSHOOT, false );
  
  const Animation& lightAnim = GameAnims::findAnimation( AnimId::KRifleShotLight );
  if ( Settings::shootingLights == true ) {
    AnimPlayer::spawn( lightAnim, this->position(), 0 );
  }
}


};  // end of namespace
